### SEIR parameter studies

# Runs an SEIR experiment to determine how maximum infected and final at 
# risk fractions depend on a parameter value
#
# Uses seir_sim (below)
#
# User specifies a list of values for one of the key parameters:
#   R0 is the basic reproductive number
#   TL is the mean incubation time
#   TI is the mean infectious duration
#   I0 is the initial infectious fraction
#   V is the initial immune fraction
#   delta is a contact factor that reduces infectivity
#
# The program is designed so that only a few lines need to be modified to
# make a new experiment (see '###' comments)
#   lines ###1 define the domain of independent variable values
# 	first and last are the min and max values for the independent variable
# 	N is the number of points (not subdivisions)
#   line ###2 identifies the x axis label for the graph
#   line ###3 links the independent variable name and values
#
# Output figure:
#   top left panel: max infectious fraction
#   top right panel: day of max infected fraction
#   bottom left panel: final fraction susceptibles
#   bottom right panel: day when end condition is reached
#
# Y contains one row for each value of the independent variable
# Columns are
#   1. maximum infectious fraction
#   2. day of max new infections
#   3. final susceptible fraction
#   4. days until I<target
#
# by Glenn Ledder
# written 2020/07/19
# revised 2020/10/27
#
# direct comments to gledder@unl.edu

### SEIR_paramstudy appears below!

#########################################

seir_sim <- function(beta,eta,gamma,E0,I0,V,target)
	{

	## DATA

	maxdays = 1000

	## FUNCTION FOR THE DIFFERENTIAL EQUATION

    	yprime <- function(yy)
		{
      	S = yy[1]
      	E = yy[2]
		I = yy[3]
      	Sp = -beta*S*I
      	Ep = -Sp-eta*E
		Ip = eta*E-gamma*I
      	yp = c(Sp,Ep,Ip)
		return(yp)
		}

	## INITIALIZATION

	# set up results data structure with Y=[S,E,I] and R

	results = matrix(0,nrow=maxdays+1,ncol=4)	
	S0 = 1-E0-I0-V
	Y = c(S0,E0,I0)
	R = V
	results[1,] = c(Y,R)

	oldI = I0

	## FUNCTION FOR rk4

      # y0 is a column vector of initial conditions at t
      # y is a column vector of values at t+dt

	rk4 <- function(dt,y0)
		{
		k1 = yprime(y0)
        	k2 = yprime(y0+0.5*dt*k1);
        	k3 = yprime(y0+0.5*dt*k2);
        	k4 = yprime(y0+dt*k3);
        	yy = y0+dt*(k1+2*k2+2*k3+k4)/6;
		return(yy)
		}

	## COMPUTATION

	for (t in 1:maxdays)
		{
		Y = rk4(1,Y)
		R = 1-sum(Y)
    		results[t+1,] = c(Y,R)
   		if (Y[2]+Y[3]>min(target,oldI))
			{
        		oldI = Y[2]+Y[3]
			} else {
	      	results = results[1:(t+1),]
        		break
			}
		}

	## OUTPUT

	return(results)

	## END

	}

#########################################

### SEIR_paramstudy starts here! ###

## DEFAULT SCENARIO DATA

R0 = 2.5;
TL = 5;
TI = 10;
I0 = 0.001;
V = 0
delta = 1

## INDEPENDENT VARIABLE DATA

### This section needs to be modified for each experiment.

###1 first and last are the min and max values for the independent variable
###1 N is the number of points (one more than the number of subdivisions)
first = 0
last = 5
N = 51

###2 xname is the name for the x axis label
xname = 'R0'

## COMMON DATA

target = 0.001
colors = c('#1f77b4','#d62728','#ff7f0e','#9467bd','#2ca02c','#17becf','#8c564b','#7f7f7f','#bcbd22','#e377c2')

## INITIALIZATION

par(mfrow=c(2,2))    # set the plotting area into a 2x2 array

dx = (last-first)/(N-1)
xvals = seq(from=first, to=last, by=dx)

# rows are for each n
Y = matrix(0, nrow=N, ncol=4);

ylabs = c('max fraction infected','days for max','final fraction susceptible','days to end condition')

## COMPUTATION 

for (n in 1:N)
	{

    	###3 The left side of this statement needs to be the independent
    	###3 variable for the experiment.
    	R0 = xvals[n]

    	eta = 1/TL
    	gamma = 1/TI
    	beta = gamma*R0

	b = eta+gamma
	c = eta*(gamma-beta*(1-V))
    	lambda = 0.5*(sqrt(b^2-4*c)-b)
    	rho = (lambda+gamma)/eta
    	E0 = rho*I0

	results = seir_sim(delta*beta,eta,gamma,E0,I0,V,target)
	S = results[,1]
	I = results[,2]
	days = length(S)-1

	Y[n,1] = max(I)
	Y[n,2] = which.max(I)-1
	Y[n,3] = results[days+1,1]
	Y[n,4] = days

	}

## OUTPUT

for (k in 1:4)
	{
	plot(xvals, Y[,k], type="l", col=colors[1], xlab=xname, ylab=ylabs[k], lwd=2, xaxs="i", yaxs="i")
	}

